import { GoogleGenAI } from "@google/genai";
import { NextResponse } from "next/server";

// Models and pricing
const MODELS = {
  RESEARCH: 'gemini-2.5-flash',
  WRITING: 'gemini-3-pro-preview',
  IMAGE: 'gemini-2.5-flash-image',
};

const PRICING = {
  GEMINI_2_5_FLASH: {
    inputPer1M: 0.075,
    outputPer1M: 0.30,
  },
  GEMINI_3_PRO: {
    inputPer1M: 1.25,
    outputPer1M: 5.00,
  },
  SEARCH_GROUNDING: {
    perQuery: 0.035,
  },
  IMAGE_GENERATION: {
    perImage: 0.04,
  },
};

// Estimate tokens (4 characters approx 1 token)
const countTokens = (text: string) => Math.ceil(text.length / 4);

interface TopicResult {
  title: string;
  context: string;
  sourceUrl: string;
  sourceTitle: string;
}

export async function POST(request: Request) {
  try {
    const { postCount = 2 } = await request.json();

    const apiKey = process.env.GEMINI_API_KEY;
    if (!apiKey) {
      return NextResponse.json(
        { error: "GEMINI_API_KEY not configured" },
        { status: 500 }
      );
    }

    const ai = new GoogleGenAI({ apiKey });

    const researchCost = { model: MODELS.RESEARCH, inputTokens: 0, outputTokens: 0, searchQueries: 1, estimatedCost: 0 };
    const writingCost = { model: MODELS.WRITING, inputTokens: 0, outputTokens: 0, estimatedCost: 0 };
    const imageCost = { model: MODELS.IMAGE, imageCount: 0, estimatedCost: 0 };

    // Step 1: Research with Google Search grounding
    const researchPrompt = `
      Find ${postCount} distinct news stories, regulatory updates, or market trends relevant to an ecommerce operations brand called "EcomFlow" (specializing in cash flow, inventory, and automation).

      CRITERIA:
      - The news must be from roughly 1 to 2 months ago (not today's breaking news, but recent history to establish authority).
      - Focus on: Supply chain regulations, Fintech updates for merchants, or major platform changes (Shopify/Amazon policies).
      - Return a list of ${postCount} distinct topics.

      Output Format:
      Return a valid JSON array of objects. Each object must have:
      - "title": A concise topic title.
      - "context": A 2-sentence summary of the news/trend.

      Do NOT include markdown formatting (like \`\`\`json). Just return the raw JSON array.
    `;

    researchCost.inputTokens += countTokens(researchPrompt);

    const researchResponse = await ai.models.generateContent({
      model: MODELS.RESEARCH,
      contents: researchPrompt,
      config: {
        tools: [{ googleSearch: {} }],
      },
    });

    const researchText = researchResponse.text || "[]";
    researchCost.outputTokens += countTokens(researchText);

    // Extract grounding metadata for real URLs
    const chunks = researchResponse.candidates?.[0]?.groundingMetadata?.groundingChunks || [];

    let topics: TopicResult[] = [];

    try {
      const cleanJson = researchText.replace(/```json/g, '').replace(/```/g, '').trim();
      const rawTopics = JSON.parse(cleanJson);

      const urls = chunks
        .map((c: any) => c.web?.uri)
        .filter((u: any): u is string => !!u);

      const titles = chunks
        .map((c: any) => c.web?.title)
        .filter((t: any): t is string => !!t);

      topics = rawTopics.map((t: any, i: number) => ({
        title: t.title,
        context: t.context,
        sourceUrl: urls[i % urls.length] || '',
        sourceTitle: titles[i % titles.length] || 'Industry Source'
      }));
    } catch {
      // Fallback topics
      topics = [
        { title: "Supply Chain Resilience", context: "Updates on global shipping logistics.", sourceUrl: "https://google.com", sourceTitle: "Google Search" },
        { title: "Ecommerce Tax Compliance", context: "New regulatory frameworks for cross-border sales.", sourceUrl: "https://google.com", sourceTitle: "Google Search" },
      ];
    }

    topics = topics.slice(0, postCount);

    // Step 2: Write blog posts
    const posts = [];

    for (const topic of topics) {
      const writingPrompt = `
        Write a professional, authoritative blog post for the brand "EcomFlow" about this topic: "${topic.title}".
        Context: ${topic.context}

        Brand Voice: Authoritative, helpful, expert, calm.
        Structure:
        - Engaging Title
        - Introduction (Hook)
        - Key Impacts on Merchants
        - Strategic Advice (How EcomFlow helps indirectly)
        - Conclusion

        CRITICAL RULE: Do NOT use em dashes anywhere in the text. Use commas, colons, or periods instead.
        Format: Markdown. Keep it concise (300-400 words).
      `;

      writingCost.inputTokens += countTokens(writingPrompt);

      const writingResponse = await ai.models.generateContent({
        model: MODELS.WRITING,
        contents: writingPrompt,
      });

      const content = writingResponse.text || "";
      writingCost.outputTokens += countTokens(content);

      // Step 3: Generate image
      const imagePrompt = `A modern, sleek, abstract 3D composition suitable for a logistics tech blog header about: ${topic.title}.
Colors: Deep Forest Green, Neon Lime Green, White. Dark mode aesthetic.
Style: Glassmorphism, frosted glass textures with soft lighting, geometric shapes, floating elements, minimalist, futuristic, professional. No text in the image.`;

      let imageUrl = "https://picsum.photos/800/400"; // Fallback

      try {
        const imageResponse = await ai.models.generateContent({
          model: MODELS.IMAGE,
          contents: {
            parts: [{ text: imagePrompt }]
          }
        });

        const parts = imageResponse.candidates?.[0]?.content?.parts || [];
        for (const part of parts) {
          if (part.inlineData && part.inlineData.data) {
            imageUrl = `data:${part.inlineData.mimeType};base64,${part.inlineData.data}`;
            imageCost.imageCount++;
            break;
          }
        }
      } catch (err) {
        console.error("Image generation failed for:", topic.title, err);
      }

      posts.push({
        title: topic.title,
        summary: topic.context,
        content: content,
        sourceUrl: topic.sourceUrl,
        sourceTitle: topic.sourceTitle,
        imageUrl: imageUrl,
      });
    }

    // Calculate costs
    const calculateMetricCost = (inputTokens: number, outputTokens: number, priceIn: number, priceOut: number) => {
      return ((inputTokens / 1000000) * priceIn) + ((outputTokens / 1000000) * priceOut);
    };

    researchCost.estimatedCost = calculateMetricCost(
      researchCost.inputTokens,
      researchCost.outputTokens,
      PRICING.GEMINI_2_5_FLASH.inputPer1M,
      PRICING.GEMINI_2_5_FLASH.outputPer1M
    ) + (researchCost.searchQueries * PRICING.SEARCH_GROUNDING.perQuery);

    writingCost.estimatedCost = calculateMetricCost(
      writingCost.inputTokens,
      writingCost.outputTokens,
      PRICING.GEMINI_3_PRO.inputPer1M,
      PRICING.GEMINI_3_PRO.outputPer1M
    );

    imageCost.estimatedCost = imageCost.imageCount * PRICING.IMAGE_GENERATION.perImage;

    const totalCost = researchCost.estimatedCost + writingCost.estimatedCost + imageCost.estimatedCost;

    return NextResponse.json({
      posts,
      costs: {
        research: researchCost,
        writing: writingCost,
        images: imageCost,
        total: totalCost
      }
    });

  } catch (error: any) {
    console.error("Generation error:", error);
    return NextResponse.json(
      { error: error.message || "Failed to generate content" },
      { status: 500 }
    );
  }
}
